/* #------------------------------------------------------------------------#
   |                                                                        |
   |   STARS.C                                                              |
   |                                                                        |
   |   Star commands for NWClient.                                          |
   |                                                                        |
   |   Copyright 1999, Frank A. Vorstenbosch.                               |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* $Id: stars.c,v 1.7 2001/12/22 12:19:51 frank Exp $ */

#include "nwclient.h"

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Star commands.                                                       |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#pragma On(Check_ltorg)

struct kernel_error *StarAttach(const char *commandTail,int numParms,struct NWClient *nw)
{  int *r;
   struct kernel_error *err;
   struct
   {  char *tree,
           *server,
           *address;
      char buffer[100]; } result;

   Trace("\nStarAttach(");
   Trace(commandTail);
   Trace(") ");

   if((err=ParseCommand("tree/k,server,address/k",commandTail,&result,sizeof(result)))!=NULL)
      return err;
   
   if(result.server)
      r=nwclient_attach(result.server,ATTACH_SERVER,nw);
   else if(result.address)
      r=nwclient_attach(result.address,ATTACH_NODE,nw);
   else
      r=nwclient_attach(result.tree,ATTACH_TREE,nw);

   return (struct kernel_error *)(r[0]);
}

/* -------------------------------------------------------------------------- */

static struct kernel_error *DetachOrLogout(const char *commandTail,int dologout,struct NWClient *nw)
{  int r;
   struct kernel_error *err;
   struct
   {  char *name;
      int all;
      char buffer[100]; } result;
   struct Server *fs,*temp;

   if((err=ParseCommand("server,all/s",commandTail,&result,sizeof(result)))!=NULL)
      return err;
   
   if(result.all || (nw->NumServers<2 && !result.name))
   {  forList2(&nw->Servers,fs,temp)
      {  if(dologout)
            r=nwclient_logout(fs,nw);
         else
            r=nwclient_detach(fs,nw);
         if(r)
            return LookupError(r);
      }
      return NULL;
   }

   if((fs=lookup_server(result.name,NULL,nw))==NULL)
      return LookupError(NCP_SERVER_UNKNOWN);

   if(dologout)
      r=nwclient_logout(fs,nw);
   else
      r=nwclient_detach(fs,nw);
   
   return LookupError(r);
}

/* .......................................................................... */

struct kernel_error *StarDetach(const char *commandTail,int numParms,struct NWClient *nw)
{  
   #if defined DEBUG
   Trace("\n StarDetach");
   #endif
   return DetachOrLogout(commandTail,0,nw); 
}

/* .......................................................................... */

struct kernel_error *StarLogout(const char *commandTail,int numParms,struct NWClient *nw)
{  
   #if defined DEBUG
   Trace("\n StarLogout");
   #endif
   return DetachOrLogout(commandTail,1,nw); 
}

/* -------------------------------------------------------------------------- */

struct kernel_error *StarLogin(const char *commandTail,int numParms,struct NWClient *nw)
{  struct kernel_error *err;
   struct
   {  char *name,
           *password;
      char buffer[100]; } result;

   if((err=ParseCommand("user=name/a,password",commandTail,&result,sizeof(result)))!=NULL)
      return err;

   return LookupError(nwclient_login(result.name,result.password,nw));
}

/* -------------------------------------------------------------------------- */

struct kernel_error *StarSysTime(const char *commandTail,int numParms,struct NWClient *nw)
{  struct kernel_error *err;
   struct
   {  char *server;
      char buffer[100]; } result;
   struct Server *fs;

   if((err=ParseCommand("server",commandTail,&result,sizeof(result)))!=NULL)
      return err;
   
   if((fs=lookup_server(result.server,NULL,nw))==NULL)
      return LookupError(NCP_SERVER_UNKNOWN);

   return LookupError(nwclient_systime(fs,nw));
}

/* -------------------------------------------------------------------------- */

static void HexByte(char **p,int i)
{  ConvertHex2(i,*p);
   (*p)+=2;
   *(*p)++=' ';
}

static void strpad(char *buffer,const char *string,int length)
{
   while(length && *string)
   {  *buffer++=*string++;
      length--; }

   while(length--)
      *buffer++=' ';
}

/* -------------------------------------------------------------------------- */

struct kernel_error *StarMount(const char *commandTail,int numParms,struct NWClient *nw)
{  struct kernel_error *err;
   struct
   {  char *local,
           *remote;
      int flag0,
          flag1,
          flaglong;
      char buffer[100]; } result;
   int *r;

   if(!numParms)
   {  struct Mount *mt;
      char *p;

      riscos_puts("Fl Ct Local name       Remote name\r\n"
                  "-- -- ---------------- ----------------\r\n");
      forList(&nw->Mounts,mt)
      {  p=result.buffer;
         
         HexByte(&p,mt->Flags);
         HexByte(&p,mt->MountCount);
         strpad(p,mt->LocalName,17);

         p+=17;
         if(nw->NumServers!=1)
         {  strcpy(p,mt->Server->ServerName);
            p+=strlen(mt->Server->ServerName); 
            *p++='/'; }
         strcpy(p,mt->RemoteName);
         p+=strlen(mt->RemoteName);
         *p++=13;
         *p++=10;
         *p=0;

         riscos_puts(result.buffer);
      }

      return NULL;
   }

   if((err=ParseCommand("local/a,remote/a,0/s,1/s,nolongnames=nl/s",commandTail,&result,sizeof(result)))!=NULL)
      return err;

   r=nwclient_mount(result.local,result.remote,(result.flag0?MOUNT_USER0:0)|(result.flag1?MOUNT_USER1:0)|(result.flaglong?MOUNT_LONGNAMES:0),nw);
   return (struct kernel_error *)(r[0]);
}

/* -------------------------------------------------------------------------- */

struct kernel_error *StarDismount(const char *commandTail,int numParms,struct NWClient *nw)
{  struct kernel_error *err;
   struct
   {  char *local;
      char buffer[100]; } result;
   char tail[100];
   struct Mount *mt;

   if((err=ParseCommand("local=disc/a",commandTail,&result,sizeof(result)))!=NULL)
      return err;

   if((mt=RosToNWName(tail,result.local,nw,0))==NULL)
      return LookupError(NCP_PATH_NOT_LOCATABLE);
   else
      return LookupError(nwclient_dismount(mt,nw));
}

/* -------------------------------------------------------------------------- */

void puti(int i)
{  char buffer[16];

   ConvertCardinal2(i,buffer);
   riscos_puts(buffer);
}

void putx3(int i)
{  char buffer[16];

   ConvertHex4(i,buffer);
   riscos_puts(buffer+1);
}

/* .......................................................................... */

struct kernel_error *StarWhoami(const char *commandTail,int numParms,struct NWClient *nw)
{  int r=0;
   #if defined DEBUG
   Trace("\n StarWhoami");
   #endif
   struct Server *fs;

   if(!nw->NumServers)
   {  riscos_puts("You are not attached to any file server.\r\n");
      return NULL; }

   forList(&nw->Servers,fs)
   {  if(fs->Flags&SERVER_LOGGED_IN)
      {
         riscos_puts("You are user ");
         riscos_puts(fs->UserName);
         riscos_puts(" logged in to connection ");
         puti(fs->ConnLow|(fs->ConnHigh<<8));
         riscos_puts(" on server ");
         riscos_puts(fs->ServerName);
         riscos_puts(".\r\n");
      }
      else
      {  riscos_puts("You are attached to server ");
         riscos_puts(fs->ServerName);
         riscos_puts(" as connection ");
         puti(fs->ConnLow|(fs->ConnHigh<<8));

         riscos_puts(" but not logged in.\r\n");
      }
   }

exit:
   return LookupError(r);
}

/* -------------------------------------------------------------------------- */

struct kernel_error *StarSetPass(const char *commandTail,int numParms,struct NWClient *nw)
{  struct kernel_error *err;
   struct
   {  char *user,
           *old,
           *new;
      char buffer[100]; 
   } result;
   int r;

   #if defined DEBUG
   Trace("\n StarSetPass");
   #endif

   if((err=ParseCommand("user/k,old,new",commandTail,&result,sizeof(result)))!=NULL)
      return err;

   r=nwclient_setpass(result.user,result.old,result.new,nw);
   if(r==NCP_PASSWORD_NOT_UNIQUE || r==NCP_PASSWORD_TOO_SHORT)
      r-=1000;

   return LookupError(r);
}

/* -------------------------------------------------------------------------- */

/* file format:
   ;type name     extensions
   faf   html     html htm
   123   ext123   extension1 extension2 extension3
   456   obey     .!boot .!run 
*/

#pragma On(Check_ltorg)

static struct kernel_error *LoadExtensions(register struct NWClient *nw,const char *buffer)
{
   int i,j,n=0;
   char *q,*strings,*next;
   const char *p;
   struct ExtensionMap *map;

   Trace("\n LoadExtensions() {");

   if((map=malloc(1<<14))==NULL || (strings=malloc(1<<14))==NULL)
   {  Trace(" nomem }");
      return &Error_NoMemory; }
   
   n=0;
   next=strings;
   p=buffer;

   while(*p)
   {  
      while(*p<' ' && *p)
         p++;

      if(*p==';' || *p=='#')
         goto next;

      q=nw->SmallBuffer;
      while(*p!=';' && *p!='#' && *p!='\n' && *p)
         *q++=*p++;

      *q=0;

      Trace(nw->SmallBuffer);

      i=0;
      q=nw->SmallBuffer;
      while((*q>='0' && *q<='9') || (*q>='A' && *q<='F') || (*q>='a' && *q<='f'))
      {  if(*q>='a')
            i=i*16+*q-'a'+10;
         else if(*q>='A')
            i=i*16+*q-'A'+10;
         else
            i=i*16+*q-'0';
         q++;
      }
      if(q==nw->SmallBuffer || (i&0xfffff000))
         goto next;

      map[n].Type=i;

      while(*q<=' ' && *q)
         q++;

      if(*q>='A' && *q<='Z')
      {  /* check whether we know this type, and if not SetVarVal it */

         i=200;
         while(*q>' ')
         {  nw->SmallBuffer[i]=*q++; 
            i++; }
         nw->SmallBuffer[i]=0;

         ConvertHex4(map[n].Type,nw->SmallBuffer+110-1);
         memcpy(nw->SmallBuffer+100,"File$Type_",10);
         if((j=ReadVarVal(nw->SmallBuffer+100,nw->SmallBuffer,16,0))==0)
            SetVarVal(nw->SmallBuffer+100,nw->SmallBuffer+200,i,0);
      }
      else
      {  while(*q>' ')
            q++; 
      }

      while(*q<=' ' && *q)
         q++;

      while(*q)
      {  
         map[n].Name=next;
         while(*q>' ')
            *next++=*q++;
         if(map[n].Name!=next)
         {
            *next++=0;

            for(i=0;i<n;i++)
            {  if(!strcmp(map[i].Name,map[n].Name))
               {  /* found duplicate, now check type */
                  if(map[i].Type==map[n].Type)
                     break;
                  else
                  {  free(strings);
                     free(map);
                     Trace(" exists }");
                     return &Error_Exists; }
               }
            }

            if(i>=n)
            {  n++;
               map[n].Type=map[n-1].Type; }
         }

         while(*q<=' ' && *q)
            q++;
      }
      
next: while(*p!='\n' && *p)
         p++;
   }

   i=next-strings;

   if(nw->ExtensionMap)
      free(nw->ExtensionMap);

   if((nw->ExtensionMap=malloc((n+1)*sizeof(struct ExtensionMap)+i))!=NULL)
   {  
      next=((char *)nw->ExtensionMap)+(n+1)*sizeof(struct ExtensionMap);
      for(i=0;i<n;i++)
      {  nw->ExtensionMap[i].Type=0xfff00000|(map[i].Type<<8);
         nw->ExtensionMap[i].Name=next;
         p=map[i].Name;
         while(*p)
         {  if(*p>='A' && *p<='Z')
               *next++=*p+32;
            else
               *next++=*p;
            p++;
         }
         *next++=0;
      }
      nw->ExtensionMap[n].Type=0;
   }

   free(strings);
   free(map);

   Trace(" }");

   return NULL;
}

#pragma Pop(Check_ltorg)

/* -------------------------------------------------------------------------- */

void DefaultExtensions(register struct NWClient *nw)
{  LoadExtensions(nw,"695 GIF gif\n"
                     "ADF Acrobat pdf\n"
                     "BF8 MPEG mpg mpeg\n"
                     "C46 Tar tar tgz\n"
                     "C85 JPEG jpg jpeg\n"
                     "DDC Archive arc gz zip\n"
                     "FAF HTML htm html\n"
                     "FB1 Wave wav\n"
                     "FB2 AVI avi\n"
                     "FE4 DOS bat com dll exe\n"
                     "FEB Obey .!Boot .!Run\n"
                     "FEC Template tpl .Template\n"
                     "FF0 TIFF tif tiff\n"
                     "FF5 PoScript eps ps\n"
                     "FF9 Sprite spr .!Sprites\n"
                     "FFA Module m\n"
                     "FFB BASIC bbc\n"
                     "FFD Data bin dat\n");
}

/* -------------------------------------------------------------------------- */

struct kernel_error *StarExtensions(const char *commandTail,int numParms,register struct NWClient *nw)
{  struct kernel_error *err;
   struct
   {  char *file;
      char buffer[256]; } result;
   int handle,i,j,n=0;
   char *buffer;

   #if defined DEBUG
   Trace("\n StarExtensions");
   #endif

   if(!numParms)
   {  riscos_puts("; NWClient file extension to RiscOS type mapping\r\n");
      if(nw->ExtensionMap)
      {  for(n=0;n<0x1000;n++)
         {  for(i=j=0;nw->ExtensionMap[i].Type;i++)
            {  if(((nw->ExtensionMap[i].Type>>8)&0xfff)==n)
               {  if(!j)
                  {  putx3(n);
                     riscos_puts(" ");
                     ConvertHex4(n,nw->SmallBuffer+110-1);
                     memcpy(nw->SmallBuffer+100,"File$Type_",10);
                     if((j=ReadVarVal(nw->SmallBuffer+100,nw->SmallBuffer,16,0))==0)
                     {  riscos_puts("&");
                        putx3(n);
                        riscos_puts("         "); }
                     else
                     {  while(j<12)
                           nw->SmallBuffer[j++]=' ';
                        nw->SmallBuffer[j++]=' ';
                        nw->SmallBuffer[j]=0;
                        riscos_puts(nw->SmallBuffer); }
                     j=1;
                  }
                  else
                     riscos_puts(" ");
                  riscos_puts(nw->ExtensionMap[i].Name);
            }  }
            if(j)
               riscos_puts("\r\n");
      }  }
      return NULL;
   }
   
   if((err=ParseCommand("file/a",commandTail,&result,sizeof(result)))!=NULL)
      return err;

   if(!strcmp(result.file,"none"))
   {
      if(nw->ExtensionMap)
         free(nw->ExtensionMap);
      nw->ExtensionMap=0;
      riscos_puts("Extension table cleared.\r\n");
      return NULL;
   }
   
   if(!strcmp(result.file,"default"))
   {
      DefaultExtensions(nw);     
      riscos_puts("Extension table reset to default.\r\n");
      return NULL;
   }

   if((handle=riscos_open(result.file,RO_RDONLY))==0)
      return &Error_FileOpen;

   if((buffer=malloc(1<<15))==NULL)
      return &Error_NoMemory;
   
   if((i=riscos_read(handle,buffer,(1<<15)-1))==0)
      return &Error_ReadError;

   buffer[i]=0;
   riscos_close(handle);

   err=LoadExtensions(nw,buffer);
   free(buffer);

   if(!err)
      riscos_puts("New extensions loaded OK.\r\n");

   return err;
}


/* ----- EOF STARS.C ----- */

