/* #------------------------------------------------------------------------#
   |                                                                        |
   |   ROUTER.C                                                             |
   |                                                                        |
   |   Router information routines for RiscOS.                              |
   |                                                                        |
   |   Copyright 1998-99, Frank A. Vorstenbosch.                            |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* $Id: router.c,v 0.16 2001/12/22 12:19:51 frank Exp $ */

#include "iriscipx.h"

#define VERSION 0x00022

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Prototypes for static functions.                                     |
   |                                                                        |
   +------------------------------------------------------------------------+ */

static void ipxrtr_del_route(struct Route *rt,register struct RiscIPX *ri);


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   RIP handler.                                                         |
   |                                                                        |
   +------------------------------------------------------------------------+ */

static int ipxrtr_riphandler(struct mbuf *mb,struct UnpackedHeader *header,void *riptr)
{  register struct RiscIPX *ri=riptr;
   struct Route *rt;
   int i;
   struct
   {  long network;
      short hops; } entry;
   unsigned short operation;
   #ifdef ALLNETS
   char flag,more;
   #endif

   #if defined DEBUG && (DEBUG&2)
   Trace("\n rip");
   #endif

   if(!memcmp(
              #ifdef ECONET
              mb->type==FRAMETYPE_ECONET?ri->EconetNode:
              #endif
              ri->Node,header->Destination.Node,IPX_NODE_LEN))
      ipxrtr_set_local_net(mb->type,header->Destination.Network,ri);

   #ifdef ALLNETS
   more=ri->MoreRoutes;
   #endif

   /* 4.x uses IPX_PT_RIP, but 3.x and presumably 2.x use type 0 */

   if((header->PacketType==IPX_PT_RIP ||  /* 4.x servers do it right          */
       header->PacketType==0) &&          /* 3.x servers do it old-fashioned  */
      (i=UnpackN(mb->next,&operation,"W",NULL,mb->pkthdr.len))>0 && operation==2)
   {  mb->pkthdr.len+=i;

      Trace(" goodop");

      while((i=UnpackN(mb->next,&entry,"LWzz",NULL,mb->pkthdr.len))>0)
      {  mb->pkthdr.len+=i;

         TraceVal(entry.network);

         i=entry.hops>254?254:(entry.hops?entry.hops:1);

         #ifdef ALLNETS
         flag=more;
         #endif
         forList(&ri->Routes,rt)
            if(rt->Network==entry.network)
            {
               #ifdef ALLNETS
               flag=1;
               #endif
               if(i<=rt->Hops)
               {
                  #if defined DEBUG && (DEBUG&2)
                  Trace("\nset route to");
                  TraceVal(entry.network);
                  #endif
                  rt->Hops=i;
                  memcpy_node(rt->Node,header->Source.Node);
                  rt->Flags=mb->type&ROUTE_FRAMEMASK;
               }
            }
         #ifdef ALLNETS
         if(!flag)
         {
            forList(&ri->Routes,rt)
               if(!rt->Network && rt->Hops)
               {  
                  #if defined DEBUG && (DEBUG&2)
                  Trace("\nadd route to");
                  TraceVal(entry.network);
                  #endif
                  rt->Network=entry.network;
                  rt->Hops=i;
                  memcpy_node(rt->Node,header->Source.Node);
                  rt->Flags=mb->type&ROUTE_FRAMEMASK;
                  flag=1;
                  break;
               }
            if(!flag)
               ri->MoreRoutes++;
         }
         #endif
      } /* for all networks listed in RIP packet */
   }

#ifdef ALLNETS
   if(!more && ri->MoreRoutes)
      AddCallBack(ri);
#endif

   ri->mbctl.freem(&ri->mbctl,mb);

   #if defined DEBUG && (DEBUG&2)
   Trace("\n endrip");
   #endif

   return 0;
}

/* -------------------------------------------------------------------------- */

#ifdef ALLNETS
void ipxrtr_callback(register struct RiscIPX *ri)
{  struct Route *rt;

   while(ri->MoreRoutes)
   {  ri->MoreRoutes--;
      if((rt=ipxrtr_add_route(0xffffffff,NULL,ri))!=NULL)
      {  TraceVal((int)rt);
         rt->Flags=ROUTE_UNKNOWN;
         rt->Network=0; }
   }
}
#endif


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Init and exit functions.                                             |
   |                                                                        |
   |   The init function is called before the network interfaces have       |
   |   been initialized, and therefore can't access the network.            |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct kernel_error *StarRoute(const char *commandTail,int numParms,struct RiscIPX *ri);

int ipxrtr_init(register struct RiscIPX *ri)
{  struct Address con;
   const struct Address *tempaddr;
   int r;

   #if defined DEBUG
   Trace("\n ipxrtr_init() {");
   #endif
   
   memcpy_address(&con,tempaddr=&BroadcastAddress);

   NewList(&ri->Routes);
   ri->MoreRoutes=0;

   /* .......................................................................... */

   if((ri->RipSocket=ipx_socket(ri))==NULL)
   {  Trace(" no socket }");
      return IPX_NO_SOCKET; }

   con.Port=IPX_SKT_RIP;
   if((r=ipx_bind(ri->RipSocket,0,IPX_SKT_RIP,ri))!=0 ||                 /* we accept any packet type */
      ((r=ipx_connect(ri->RipSocket,IPX_PT_RIP,&con,ri))!=0))  /* but we use PT_RIP on send */
   {  ipx_close(ri->RipSocket,ri);
      Trace(" bind/connect err }");
      return r; }

   ipx_recv_handler(ri->RipSocket,&ipxrtr_riphandler,ri,ri);

   /* .......................................................................... */

   Trace(" }");
   return 0;
}

/* -------------------------------------------------------------------------- */

int ipxrtr_exit(register struct RiscIPX *ri)
{  struct Route *rt,*next;

   #if defined DEBUG
   Trace("\n ipxrtr_exit ");
   #endif
   
#ifdef ALLNETS
   if(ri->MoreRoutes)
      RemoveCallBack(ri);
#endif

   ipx_close(ri->RipSocket,ri);

   forList2(&ri->Routes,rt,next)
      ipxrtr_del_route(rt,ri);

   return 0;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   User routing functions.                                              |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int ipxrtr_set_local_net(int frameType,long network,register struct RiscIPX *ri)
{  struct Route *rt;
   int r=IPX_NO_ROUTE_TO_NETWORK;

   #if defined DEBUG && (DEBUG&2)
   Trace("\n set_local_net");
   TraceByte(frameType);
   TraceVal(network);
   #endif

   forList(&ri->Routes,rt)
   {
      if(!rt->Hops && (rt->Flags&ROUTE_FRAMEMASK)==frameType)
      {  rt->Network=network;
         rt->Flags&=~ROUTE_UNKNOWN;
         r=0; }

      if(rt->Hops && rt->Network==network)
      {  rt->Network=0;
         rt->Hops=255;
         rt->Flags|=ROUTE_UNKNOWN; }
   }

   return r;
}

/* -------------------------------------------------------------------------- */

struct Route *ipxrtr_get_local_net(int frameType,register struct RiscIPX *ri)
{  struct Route *rt;
   int i=0;

   #if defined DEBUG && (DEBUG&2)
   Trace("\n get_local_net");
   TraceByte(frameType);
   #endif

   forList(&ri->Routes,rt)
      if(!rt->Hops && (rt->Flags&ROUTE_FRAMEMASK)==frameType)
      {
         if(i>1)
         {  Disable(ri);
            RemoveNode(rt);
            AddHead(&ri->Routes,rt);
            Enable(ri);
         }
         
         return rt;
      }
      else
         i++;

   return NULL;
}

/* -------------------------------------------------------------------------- */

long ipxrtr_get_local_net_number(int frameType,register struct RiscIPX *ri)
{  struct Route *rt;

   rt=ipxrtr_get_local_net(frameType,ri);
   if(rt && !(rt->Flags&ROUTE_UNKNOWN))
      return rt->Network;

   return 0L;
}

/* -------------------------------------------------------------------------- */

static struct Route *ipxrtr_find(long network,register struct RiscIPX *ri)
{  struct Route *rt;
   int i=0;

   #if defined DEBUG && (DEBUG&2)
   Trace("\n ipxrtr_find(");
   TraceVal(network);
   Trace(")");
   #endif

   if(!network)
      return NULL;

   Trace(" {");

   forList(&ri->Routes,rt)
      if(rt->Network==network)
      {
         if(i>1)
         {  struct Route *t;
            Disable(ri);

            RemoveNode(rt);
            FirstNode(&ri->Routes,t);
            InsertNode(t,rt);

            Enable(ri);
         }
         
         Trace(" }");
         return rt;
      }
      else
         i++;

   Trace(" }");
   return NULL;
}

/* -------------------------------------------------------------------------- */

#define LOOKUP_TRIES 32

struct Route *ipxrtr_lookup(long network,register struct RiscIPX *ri)
{  struct Route *rt;
   volatile int to;
   #ifdef ALLNETS
   long ffffffff=-1;
   #endif

   #if defined DEBUG && (DEBUG&2)
   Trace("\n ipxrtr_lookup(");
   TraceVal(network);
   Trace(") {");
   #endif

   switch(network)
   {  case 0xffffffff:
      case 0:
         rt=NULL;
         break;
      default:
         if((rt=ipxrtr_find(network,ri))==NULL &&
            (rt=ipxrtr_add_route(network,NULL,ri))==NULL)
         {  Trace(" }");
            return NULL; }
         if(!(rt->Flags&ROUTE_UNKNOWN))
         {  Trace(" }");
            return rt; }
   }

   for(to=0;to<LOOKUP_TRIES;to++)    /* 320 ms timeout */
   {  
      #ifdef ALLNETS
      if((to&28)==0)
         ipx_send_pack(ri->RipSocket,&ffffffff,"zk\001L4k\377",ri);
      #endif

      if((to&28)==4)
         ipx_send_pack(ri->RipSocket,&network,"zk\001L4k\377",ri);

      sleep10ms();
      
      if(rt && !(rt->Flags&ROUTE_UNKNOWN))
      {  Trace(" }");
         return rt; }
   }

   if(rt)
   {
      if(!rt->Hops)  /* special case for local network, we never delete that */
      {  Trace(" }");
         return rt; }

      ipxrtr_del_route(rt,ri);
   }

   Trace(" }");
   return NULL;
}

/* -------------------------------------------------------------------------- */

struct Route *ipxrtr_add_route(long network,char *node,register struct RiscIPX *ri)
{  struct Route *rt;
   int new=0;

   #if defined DEBUG && (DEBUG&2)
   Trace("\n ipxrtr_add_route(");
   TraceVal(network);
   TraceVal((int)node);
   Trace(") {");
   #endif

   if((rt=ipxrtr_find(network,ri))==NULL)
   {  if((rt=malloc(sizeof(struct Route)))==NULL)
      {  Trace(" }");
         return NULL; }
      new=1;
   }

   TraceVal((int)rt);

   rt->Network=network;

   if(node)
   {  rt->Flags=0;
      rt->Hops=255;
      memcpy_node(rt->Node,node); }
   else
   {  if(network)
         rt->Hops=255;
      else
         rt->Hops=0;
      rt->Flags=ROUTE_UNKNOWN; }

   if(new)
   {  Disable(ri);
      AddTail(&ri->Routes,rt);
      Enable(ri); }

   Trace(" }");

   return rt;
}

/* -------------------------------------------------------------------------- */

static void ipxrtr_del_route(struct Route *rt,register struct RiscIPX *ri)
{
   #if defined DEBUG && (DEBUG&2)
   Trace("\n ipxrtr_del_route");
   TraceVal((int)rt);
   #endif

   Disable(ri);
   RemoveNode(rt);
   Enable(ri);

   free(rt);
}

/* -------------------------------------------------------------------------- */

int ipxrtr_delete(long network,register struct RiscIPX *ri)
{  struct Route *rt;

   #if defined DEBUG && (DEBUG&2)
   Trace("\n ipxrtr_delete");
   TraceVal(network);
   #endif

   if((rt=ipxrtr_find(network,ri))==NULL)
      return -1;

   if(!rt->Hops)
   {  rt->Network=0;
      rt->Flags|=ROUTE_UNKNOWN; }
   else
      ipxrtr_del_route(rt,ri);

   return 0;
}

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Star commands.                                                       |
   |                                                                        |
   +------------------------------------------------------------------------+ */

void putlong(int i)
{  char buffer[16];

   ConvertHex8(i,buffer);
   riscos_puts(buffer);
}

void putbyte(int i)
{  char buffer[16];

   ConvertHex8(i,buffer);
   riscos_puts(buffer+6);
}

void putshort(int i)
{  char buffer[16];

   ConvertHex8(i,buffer);
   riscos_puts(buffer+4);
}

void putaddress(char *address)
{  char buffer[16];
   int i;

   i=((address[0]&255)<<24)|((address[1]&255)<<16)|((address[2]&255)<<8)|((address[3]&255));
   ConvertHex8(i,buffer);
   riscos_puts(buffer);
   i=((address[4]&255)<<8)|((address[5]&255));
   ConvertHex8(i,buffer);
   riscos_puts(buffer+4);
}

struct kernel_error *StarRoute(const char *commandTail,int numParms,register struct RiscIPX *ri)
{  struct Route *rt;
   extern void PutsType(int type);

   #ifdef ALLNETS
   ipxrtr_lookup(0xffffffff,ri);
   #endif

   riscos_puts("Network  Node        Hop Fl Frame\r\n"
               "=================================\r\n");
   forList(&ri->Routes,rt)
   {
      if(!rt->Network)
         continue;

      putlong(rt->Network);
      riscos_puts(" ");
      putaddress(rt->Node);
      riscos_puts(" ");
      putbyte(rt->Hops);
      riscos_puts(" ");
      putbyte(rt->Flags);
      riscos_puts(" ");
      PutsType(rt->Flags&ROUTE_FRAMEMASK);

      riscos_puts("\r\n");
   }

   #ifdef ALLNETS
   ipxrtr_lookup(0xffffffff,ri);
   #endif

   return NULL;
}

/* -------------------------------------------------------------------------- */

int *ipxrtr_routelist(struct Route *rt,register struct RiscIPX *ri)
{  struct Route *tmp;

   ri->ReturnStruct[0]=ri->ReturnStruct[1]=0;

   if(!rt)
   {  if(!FirstNode(&ri->Routes,rt))
         return ri->ReturnStruct;
   }
   else
   {
      int found=0;
      forList(&ri->Routes,tmp)
      {  if(tmp==rt)
         {  found=1;
            NextNode(rt);
            break; }
      }
      if(!found)
      {  ri->ReturnStruct[1]=1;
         return ri->ReturnStruct; }
   }

   if(IsTailNode(rt))
      return ri->ReturnStruct;
   
   ri->ReturnStruct[0]=(int)rt;
   ri->ReturnStruct[1]=rt->Network;
   ri->ReturnStruct[2]=rt->Hops;
   ri->ReturnStruct[3]=rt->Flags;
   ri->ReturnStruct[4]=(int)&rt->Node;

   return ri->ReturnStruct;
}

/* ----- EOF ----- */
