/* ͻ
                                                                        
      LISTS.H                                                           
                                                                        
      Improved general purpose list functions.                          
      These now have much fewer casts, and should be type-safe.         
                                                                        
      Copyright 1998, Frank A. Vorstenbosch                             
                                                                        
   ͼ

   File created 5-dec-98 */

#ifndef __LISTS_H
#define __LISTS_H

struct __Node
{
   struct __Node *__Next;
   struct __Node *__Prev;
};

typedef struct __List
{  
   union
   {  struct
      {  struct __Node __Node;
         void *__dummy; } __Head;
      struct
      {  void *__dummy;
         struct __Node __Node; } __Tail;
   } __The;
} LIST;

#define NODE struct __Node __Links

/*  macros to operate on lists  */

/* Initialize a list header before the first use */
#define NewList(l) do { (l)->__The.__Head.__Node.__Next=&(l)->__The.__Tail.__Node; \
                        (l)->__The.__Tail.__Node.__Prev=&(l)->__The.__Head.__Node; \
                        (l)->__The.__Tail.__Node.__Next=(struct __Node *)0; } while(0)

/* Add a new node to the head of the list */
#define AddHead(l,n) do {  (n)->__Links.__Next=(l)->__The.__Head.__Node.__Next; \
                           (l)->__The.__Head.__Node.__Next=(n)->__Links.__Next->__Prev=&((n)->__Links); \
                           (n)->__Links.__Prev=&(l)->__The.__Head.__Node; } while(0)

/* Add a new node to the tail of the list */
#define AddTail(l,n) do {  (n)->__Links.__Prev=(l)->__The.__Tail.__Node.__Prev; \
                           (l)->__The.__Tail.__Node.__Prev=(n)->__Links.__Prev->__Next=&((n)->__Links); \
                           (n)->__Links.__Next=&(l)->__The.__Tail.__Node; } while(0)

/* Remove the head node of the list, evaluates to nonzero if the head was removed, or zero if the list was empty */
#define RemoveHead(l)   (((l)->__The.__Head.__Node.__Next->__Next) && \
                           ((l)->__The.__Head.__Node.__Next=(l)->__The.__Head.__Node.__Next->__Next, \
                           (l)->__The.__Head.__Node.__Next->__Prev=&(l)->__The.__Head.__Node,1))

/* Remove the tail node of the list, evaluates to nonzero if the tail was removed, or zero if the list was empty */
#define RemoveTail(l)   (((l)->__The.__Tail.__Node.__Prev->__Prev) && \
                           ((l)->__The.__Tail.__Node.__Prev=(l)->__The.__Tail.__Node.__Prev->__Prev, \
                           (l)->__The.__Tail.__Node.__Prev->__Next=&(l)->__The.__Tail.__Node,1))

/* Insert new node n in the list following node p */
#define InsertNode(p,n) do {  (n)->__Links.__Next=(p)->__Links.__Next; \
                              (p)->__Links.__Next=((n)->__Links.__Next)->__Prev=&((n)->__Links); \
                              (n)->__Links.__Prev=&((p)->__Links); } while(0)

/* Remove a node from the list */
#define RemoveNode(n)   do {  (n)->__Links.__Prev->__Next=(n)->__Links.__Next; \
                              (n)->__Links.__Next->__Prev=(n)->__Links.__Prev; } while(0)


/*  macros to step through the nodes in a list  */

/* Point node pointer to the (virtual) head node of the list */
#define HeadNode(l,n)      (((struct __Node *)(n))=&(l)->__The.__Head.__Node)

/* Point node pointer to the (virtual) tail node of the list */
#define TailNode(l,n)      (((struct __Node *)(n))=&(l)->__The.__Tail.__Node)
                                    
/* Point node pointer to the first node in the list, evaluates to zero if the list is empty */
#define FirstNode(l,n)     (((struct __Node *)(n))=(l)->__The.__Head.__Node.__Next,(n)->__Links.__Next)

/* Point node pointer to the last node in the list, evaluates to zero if the list is empty */
#define LastNode(l,n)      (((struct __Node *)(n))=(l)->__The.__Tail.__Node.__Prev,(n)->__Links.__Prev)
                                    
/* Point node pointer to the next node in the list */
#define NextNode(n)        (((struct __Node *)(n))=(n)->__Links.__Next)

/* Point node pointer to the previous node in the list */
#define PreviousNode(n)    (((struct __Node *)(n))=(n)->__Links.__Prev)
                                    
/* Step through all nodes in the list */
#define forList(l,n)       for(((struct __Node *)(n))=(l)->__The.__Head.__Node.__Next; \
                              !IsTailNode(n);NextNode(n))

/* Step through all nodes in the list using a temporary pointer p so that nodes can be removed safely */
#define forList2(l,n,p)    for(((struct __Node *)(p))=(l)->__The.__Head.__Node.__Next; \
                               ((struct __Node *)(n))=((struct __Node *)(p)),(((struct __Node *)(p))=(p)->__Links.__Next)!=0;)

/* Step through all nodes in the list in reverse order (not encouraged) */
#define tsiLrof(l,n)       for(((struct __Node *)(n))=(l)->__The.__Tail.__Node.__Prev; \
                               !IsHeadNode(n);PrevNode(n))


/*  macros to test a list or node  */

/* Evaluates to nonzero if the list is empty */
#define IsEmptyList(l)        ((l)->__The.__Tail.__Node.__Prev==&(l)->__The.__Head.__Node)

/* Evaluates to nonzero if this node is the last node in the list */
#define IsLastNode(n)         (IsTailNode(n)?0:!((n)->__Links.__Next->__Next))

/* Evaluates to nonzero if this node is the fist node in the list */
#define IsFirstNode(n)        (IsHeadNode(n)?0:!((n)->__Links.__Prev->__Prev))

/* Evaluates to nonzero if this node is the (virtual) tail node in the list */
#define IsTailNode(n)         (!((n)->__Links.__Next))

/* Evaluates to nonzero if this node is the (virtual) head node in the list */
#define IsHeadNode(n)        (!((n)->__Links.__Prev))

#endif

/*  EOF LISTS.H  */
