/* #------------------------------------------------------------------------#
   |                                                                        |
   |   IRISCIPX.H                                                           |
   |                                                                        |
   |   Internal definitions for RiscIPX.                                    |
   |                                                                        |
   |   Copyright 1998, Frank A. Vorstenbosch                                |
   |                                                                        |
   #------------------------------------------------------------------------# */

#include "pragmas.h"
#include "../libmod.h"
#include "lists.h"
#include "errors.h"
#include "debug.h"
#define _NODEF_RISCIPX_
#include "riscipx.h"

/* -------------------------------------------------------------------------- */

#define FIRSTPORT       0x4000
#define LASTPORT        0x7fff
#ifdef ECONET
#define ECONET_IPX_PORT 'X'
#endif


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Socket.                                                              |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct UnpackedHeader;
struct RiscIPX;
struct Socket
{  NODE;
   struct Address Connection;
   struct mbuf *ReadQueue,*TailQueue;
   int (*Handler)(struct mbuf *,struct UnpackedHeader *,void *);
   void *HandlerData;
   unsigned short Port;
   char AcceptPT,
        SendPT;
   volatile char Flags;
      #define SOCKET_BOUND             (1<<0)
      #define SOCKET_CONNECT_NETWORK   (1<<1)
      #define SOCKET_CONNECT_NODE      (1<<2)
      #define SOCKET_CONNECT_PORT      (1<<3)
      #define SOCKET_ANY_TYPE          (1<<4)
};

#if 0
struct SocketSPX
{  struct Socket Socket;

   struct mbuf *Retransmit;
   unsigned short Sequence,
                  AckSeq;
   char State;
      #define SPX_STATE_CLOSED      0
      #define SPX_STATE_UNCONNECTED 1
      #define SPX_STATE_LISTEN      2
      #define SPX_STATE_CONNECTING  3
      #define SPX_STATE_ESTABLISHED 4
};
#endif


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Various IPX frame types.                                             |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct Frame_8022
{  unsigned short SAPs;       /* always 0xe0e0 for IPX over 802.2 */
   /* char Control;           -- ignore, set to 3 for transmit */
};

struct Frame_8023
{  /* empty */
};
   
struct Frame_SNAP
{  unsigned short SAPs;       /* always 0xaaaa for SNAP */
   char Control;              /* always 3 for SNAP */
   /* char OUI[3];            -- ignore, set to zero on transmit */
   unsigned short Protocol;   /* 0x8137 for IPX */
};
   
struct Frame_E2
{  unsigned short Protocol;   /* 0x8137 for IPX */
};

#ifdef ECONET
#define ECONET_PREHEADER (PACKETHEADERSIZE-2*sizeof(struct Address))
#endif

   
/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Routing Information Protocol.                                        |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct Route
{  NODE;
   long Network;
   char Node[IPX_NODE_LEN];   /* MUST be aligned to 4 byte address */
   volatile char Flags;
      #define ROUTE_UNKNOWN   (1<<7)
      #define ROUTE_FRAMEMASK 7
   char Hops;
};


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Static data.                                                         |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct RiscIPX
{
   struct mbctl mbctl;

   LIST Sockets;
   LIST Routes;

   struct Socket *RipSocket;

   int NextPort,
       SwiBase,
       Unit,
       MTU,
       ReturnStruct[8];

   #ifdef ECONET
   int RxCB;
   char EconetNode[IPX_NODE_LEN];
   struct mbuf *Listen;
   #endif
   
   char Node[IPX_NODE_LEN];

   volatile char DisableCount;

   char FrameType,
           #define FRAMETYPE_AUTO   0
           #define FRAMETYPE_8022   1
           #define FRAMETYPE_8023   2
           #define FRAMETYPE_SNAP   3
           #define FRAMETYPE_E2     4
           #ifdef ECONET
           #define FRAMETYPE_ECONET 5
           #define FRAMETYPE_LAST   5
           #else
           #define FRAMETYPE_LAST   4
           #endif
           #define FRAMETYPE_FIRST  1
        Flags,
           #define IPX_FT_CYCLE_MASK     (7)
           #ifdef ECONET
           #define IPX_REGISTERED_ECONET (1<<5)
           #endif
           #define IPX_REGISTERED_40000  (1<<6)
           #define IPX_REGISTERED_18137  (1<<7)
        MoreRoutes;
};

extern const struct Address BroadcastAddress;


/* #------------------------------------------------------------------------#
   |                                                                        |
   |   Packet structures for well-known services.                           |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Sequenced Packet Exchange.                                           |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct SPXPacket
{  char ConnectionControl,
      #define SPX_CC_NEGOTIATE   (1<<2)   /* negotiate size request/response (SPX2) */
      #define SPX_CC_SPX2TYPE    (1<<3)
      #define SPX_CC_ENDOFMSSG   (1<<4)   /* end of message */
      #define SPX_CC_ATTENTION   (1<<5)   
      #define SPX_CC_ACKREQD     (1<<6)   /* acknowledgement required   */
      #define SPX_CC_SYSTEM      (1<<7)   /* system packet              */
        DatastreamType;
      #define SPX_DT_RLSREQ      0xfc     /* orderly release request             */
      #define SPX_DT_RLSACK      0xfd     /* orderly release acknowledgement     */
      #define SPX_DT_END         0xfe     /* end of connection                   */
      #define SPX_DT_ENDACK      0xff     /* end of connection acknowledgement   */
   unsigned short SourceConnection,       /* for SPX multiplexing */
                  DestinationConnection,  /* for SPX demultiplexing */
                  SequenceNumber,
                  AcknowledgeNumber,      /* indicates the sequence number of the next packet expected */
                  AllocationNumber;       /* indicates how many receive buffers are free in the station that sent this SPX packet */
   char Data[1];
};


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Echo Protocol.                                                       |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct EchoPacket
{  unsigned short Operation;
      #define ECHO_REQUEST    1
      #define ECHO_REPLY      2
   char Data[1];
};


/* #------------------------------------------------------------------------#
   |                                                                        |
   |   Function prototypes.                                                 |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* ----- ipx.c -------------------------------------------------------------- */

extern struct RiscIPX *ipx_init(void);
extern int ipx_exit(register struct RiscIPX *ri);

extern int ipx_sizeof_struct(void);

/* ----- router.c ----------------------------------------------------------- */

extern int ipxrtr_init(register struct RiscIPX *ri);
extern int ipxrtr_exit(register struct RiscIPX *ri);
extern long ipxrtr_get_local_net_number(int frameType,register struct RiscIPX *ri);

/* ----- subipx.arm --------------------------------------------------------- */

extern void PatchSwis(int);
extern void IntHandler();
extern void IntHandlerE2();

extern struct kernel_error *Register(int flags,int unit,int frametype,int addresslevel,int errorlevel,
                                     void *privword,void (*handler)(void));
extern struct kernel_error *Transmit(int,int,struct mbuf *,char *);
extern int DCIVersion(void);
extern int GetMTU(int flags,int unit);

void putlong(int i);
void putbyte(int i);
void putshort(int i);
void putaddress(char *address);
extern void AddCallBack(struct RiscIPX *ri);
extern void RemoveCallBack(struct RiscIPX *ri);
extern char Version[4];

#ifdef ECONET
extern void EventHandler(int,int,int);
extern int Econet_DoTransmit(int flag,int port,int station,int net,
                             void *buffer,int bufsize,int count,int delay);
extern int Econet_ReadReceive(int handle,int *flag,int *port,int *station,
                              int *net,void **buffer,int *length);
extern int Econet_CreateReceive(int port,int station,int net,void *buffer,int bufsize);
extern int OSByte(int,...);
#endif

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Inline assembly functions.                                           |
   |                                                                        |
   +------------------------------------------------------------------------+ */

_Asm struct dibchain *ServiceCall9B(void)
{
   mov   %r0,#0
   mov   %r1,#0x9b
   swi   0x30     ; svccall
};

_Asm struct kernel_error *OpenSession(struct mbctl *ctl)
{  %reg ctl
   mov   %r0,ctl
   swi   0x20000|0x4a580   ; Mbuf_OpenSession
   movvc %r0,#0
};

_Asm struct kernel_error *CloseSession(struct mbctl *ctl)
{  %reg ctl
   mov   %r0,ctl
   swi   0x20000|0x4a581   ; Mbuf_CloseSession
   movvc %r0,#0
};

_Asm int _Disable(void)
{
   swi   0x14  ; disable
};
#define Disable(ri) do { if(!ri->DisableCount) _Disable(); ri->DisableCount++; } while(0)

_Asm int _Enable(void)
{
   swi   0x13  ; enable
};
#define Enable(ri) do { if(!--ri->DisableCount) _Enable(); } while(0)

_Asm void memcpy_node(char *dest,const char *src)
{  %reg dest,src
   ldmia src,{%r0,%r1}
   str   %r0,[dest]
   strb  %r1,[dest,#4]
   mov   %r1,%r1,LSR #8
   strb  %r1,[dest,#5]
};

_Asm void memcpy_address(struct Address *dest,const struct Address *src)
{  %reg dest,src
   ldmia src,{%r0,%r1,%r2}
   stmia dest,{%r0,%r1,%r2}
};


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Debugging.                                                           |
   |                                                                        |
   +------------------------------------------------------------------------+ */

/* ----- EOF ----- */

